#!/bin/bash
# BundleLive Silent Printer — Double-click to install
# Prints bin tags automatically during live shows — zero dialogs

clear
echo ""
echo "  🖨️  BundleLive Silent Printer Installer"
echo "  ========================================="
echo ""

INSTALL_DIR="$HOME/.bundlelive"
PLIST_NAME="com.bundlelive.printserver"
PLIST_PATH="$HOME/Library/LaunchAgents/$PLIST_NAME.plist"
LOG_DIR="$HOME/Library/Logs/BundleLive"
NODE_PATH=$(which node 2>/dev/null)
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"

# Check for Node.js
if [ -z "$NODE_PATH" ]; then
  echo "  ⚠️  Node.js is required but not installed."
  if command -v brew &>/dev/null; then
    echo "  📥 Installing via Homebrew..."
    brew install node
    NODE_PATH=$(which node)
  else
    echo ""
    echo "  Please install Node.js first: https://nodejs.org"
    echo "  Then double-click this file again."
    echo ""
    read -p "  Press Enter to close..."
    exit 1
  fi
fi

echo "  ✓ Node.js found: $NODE_PATH"

# Create install directory
mkdir -p "$INSTALL_DIR"
mkdir -p "$LOG_DIR"

# Copy print server from extension folder
if [ -f "$SCRIPT_DIR/print-server.js" ]; then
  cp "$SCRIPT_DIR/print-server.js" "$INSTALL_DIR/print-server.js"
  echo "  ✓ Print server installed to $INSTALL_DIR"
else
  echo "  ❌ print-server.js not found in $SCRIPT_DIR"
  read -p "  Press Enter to close..."
  exit 1
fi

# Stop existing service
launchctl unload "$PLIST_PATH" 2>/dev/null || true

# Create LaunchAgent (auto-starts on login, restarts if crashes)
cat > "$PLIST_PATH" << EOF
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0">
<dict>
    <key>Label</key>
    <string>$PLIST_NAME</string>
    <key>ProgramArguments</key>
    <array>
        <string>$NODE_PATH</string>
        <string>$INSTALL_DIR/print-server.js</string>
    </array>
    <key>RunAtLoad</key>
    <true/>
    <key>KeepAlive</key>
    <true/>
    <key>StandardOutPath</key>
    <string>$LOG_DIR/print-server.log</string>
    <key>StandardErrorPath</key>
    <string>$LOG_DIR/print-server-error.log</string>
</dict>
</plist>
EOF

launchctl load "$PLIST_PATH"

echo "  ✓ Service installed and started"

# Wait and verify
sleep 2

if curl -s http://localhost:9099/health 2>/dev/null | grep -q "ok"; then
  echo ""
  echo "  ✅ SUCCESS! Silent printing is active."
  echo ""
  echo "  • Bin tags print automatically during shows"
  echo "  • Starts automatically when you log in"  
  echo "  • No dialogs, no popups, no interruptions"
  echo "  • Prints to your default printer"
  echo ""
  echo "  To change printer: System Settings → Printers → Set Default"
  echo "  To uninstall: launchctl unload $PLIST_PATH"
else
  echo ""
  echo "  ⚠️  Server installed but health check failed."
  echo "  Check logs: cat $LOG_DIR/print-server-error.log"
  echo "  Try: $NODE_PATH $INSTALL_DIR/print-server.js"
fi

echo ""
read -p "  Press Enter to close..."
